<?php
/*
Plugin Name: BrainyPress
Plugin URI: https://BrainyPress.net
Description: The Ultimate AI Content Machine. Automate your blog with human-quality articles using AI. Features: Real-time Google Grounding (Forced), Direct Twitter Auto-Post (4 Keys), and Neubrutalism UI.
Version: 2.7
Author: Naveed Ali Shah
Author URI: https://mrnaveedshah.com
License: GPLv2 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

// ==========================================================================
// 1. ADMIN ASSETS & ULTIMATE NEUBRUTALISM UI
// ==========================================================================
add_action('admin_head', 'nf_custom_styles');

function nf_custom_styles() {
    ?>
    <style>
        /* --- VARIABLES --- */
        :root {
            --nb-lime: #ccff00;
            --nb-black: #000000;
            --nb-white: #ffffff;
            --nb-border: 4px solid #000000; /* Thicker Border */
            --nb-shadow: 6px 6px 0px #000000; /* Deeper Shadow */
            --nb-shadow-hover: 2px 2px 0px #000000;
        }

        /* --- GLOBAL --- */
        .nf-wrap {
            font-family: 'Courier New', Courier, monospace;
            max-width: 1100px;
            margin: 40px auto;
            color: var(--nb-black);
        }
        
        /* DOT GRID BACKGROUND */
        body {
            background-image: radial-gradient(#94a3b8 1.5px, transparent 1.5px);
            background-size: 24px 24px;
            background-color: #f1f5f9;
        }

        /* --- CARDS --- */
        .nf-card {
            background: var(--nb-white);
            border: var(--nb-border);
            border-radius: 16px;
            padding: 35px;
            margin-bottom: 40px;
            box-shadow: var(--nb-shadow);
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }
        .nf-card:hover {
            transform: translate(2px, 2px);
            box-shadow: var(--nb-shadow-hover);
        }

        /* --- HEADER (BOLD DESIGN) --- */
        .nf-header {
            background: var(--nb-lime);
            text-align: center;
            padding: 50px 30px;
            border: var(--nb-border);
            border-radius: 20px;
            box-shadow: 8px 8px 0px #000;
            margin-bottom: 50px;
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 15px;
            position: relative;
        }
        
        .nf-animated-title {
            font-weight: 900;
            font-size: 4em;
            text-transform: uppercase;
            color: var(--nb-black);
            margin: 0;
            letter-spacing: -3px;
            line-height: 1;
            text-shadow: 3px 3px 0px #fff;
        }

        .nf-slogan {
            font-size: 16px;
            font-weight: 800;
            letter-spacing: 1px;
            text-transform: uppercase;
            background: var(--nb-black);
            color: var(--nb-white);
            padding: 12px 30px;
            border: var(--nb-border);
            border-radius: 50px;
            display: inline-block;
            box-shadow: 4px 4px 0px rgba(255,255,255,0.8);
            transform: rotate(-1deg);
        }

        /* --- INPUTS --- */
        .nf-input, .nf-select, .nf-textarea {
            width: 100%;
            border: var(--nb-border);
            border-radius: 10px;
            padding: 16px;
            font-size: 16px;
            font-weight: bold;
            background: #fff;
            box-shadow: 3px 3px 0px rgba(0,0,0,0.1);
            margin-bottom: 15px;
            box-sizing: border-box; 
            font-family: inherit;
        }
        .nf-input:focus, .nf-textarea:focus {
            outline: none;
            background: #fff;
            box-shadow: var(--nb-shadow);
            transform: translate(-2px, -2px);
        }

        /* --- BUTTONS --- */
        .nf-btn {
            display: inline-block;
            background: var(--nb-black);
            color: var(--nb-lime);
            border: var(--nb-border);
            padding: 16px 32px;
            font-weight: 900;
            text-transform: uppercase;
            cursor: pointer;
            box-shadow: var(--nb-shadow);
            border-radius: 10px;
            text-decoration: none;
            transition: all 0.1s;
            font-size: 16px;
        }
        .nf-btn:hover {
            transform: translate(3px, 3px);
            box-shadow: 2px 2px 0px #000;
            background: #fff;
            color: var(--nb-black);
        }
        .nf-btn-danger { background: #ff4757; color: #fff; }
        .nf-btn-danger:hover { background: #ff6b81; color: #000; }
        
        .nf-btn-accent { background: #2ed573; color: #000; }

        /* --- TABLES --- */
        .nf-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0 15px;
        }
        .nf-table th { 
            text-align: left; 
            padding: 15px; 
            font-weight: 900; 
            font-size: 1.2em; 
            text-transform: uppercase;
            background: transparent;
            color: #000;
        }
        .nf-table td { 
            background: #fff; 
            border: 3px solid #000; 
            padding: 25px; 
            vertical-align: middle;
            font-weight: 600;
            font-size: 15px;
        }
        .nf-table tr td:first-child { border-top-left-radius: 12px; border-bottom-left-radius: 12px; border-right: none; }
        .nf-table tr td:last-child { border-top-right-radius: 12px; border-bottom-right-radius: 12px; border-left: none; box-shadow: 5px 5px 0px #000; }
        
        /* --- FOOTER (IMPROVED) --- */
        .nf-footer {
            margin-top: 60px; 
            padding: 40px;
            border: var(--nb-border); 
            border-radius: 16px;
            background: var(--nb-black); 
            box-shadow: var(--nb-shadow);
            text-align: center;
            color: #fff;
        }
        .nf-footer h3 { color: var(--nb-lime); margin-top: 0; font-size: 20px; }
        .nf-footer p { color: #ccc; margin-bottom: 25px; }
        
        .nf-footer-links {
            display: flex;
            justify-content: center;
            gap: 15px;
            flex-wrap: wrap;
        }
        .nf-footer-btn {
            background: #333;
            color: #fff;
            padding: 10px 20px;
            border-radius: 50px;
            text-decoration: none;
            font-weight: bold;
            border: 2px solid #fff;
            cursor: pointer;
            transition: 0.2s;
        }
        .nf-footer-btn:hover {
            background: var(--nb-lime);
            color: #000;
            border-color: #000;
            transform: translateY(-3px);
        }

        /* --- MODAL SYSTEM --- */
        #nf-info-modal {
            position: fixed; top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(0,0,0,0.9); z-index: 100000;
            display: none; justify-content: center; align-items: center;
            backdrop-filter: blur(5px);
        }
        .nf-modal-content {
            background: #fff;
            border: 5px solid #000;
            padding: 40px;
            width: 700px;
            max-width: 90%;
            max-height: 85vh;
            overflow-y: auto;
            border-radius: 20px;
            box-shadow: 15px 15px 0px var(--nb-lime);
            position: relative;
        }
        .nf-close-modal {
            position: absolute; top: 15px; right: 20px;
            font-size: 30px; font-weight: 900; cursor: pointer; color: #000;
            background: var(--nb-lime); width: 40px; height: 40px; text-align: center;
            line-height: 40px; border-radius: 50%; border: 2px solid #000;
        }
        .nf-close-modal:hover { transform: scale(1.1); }
        .nf-modal-title {
            font-size: 32px; font-weight: 900; margin-bottom: 25px;
            text-transform: uppercase; border-bottom: 5px solid var(--nb-lime);
            padding-bottom: 10px; display: inline-block;
        }
        .nf-modal-body p { font-size: 17px; line-height: 1.7; margin-bottom: 20px; color: #333; }
        .nf-modal-body ul { margin-left: 20px; font-size: 16px; }
        .nf-modal-body li { margin-bottom: 10px; }

        /* --- OVERLAY & PROGRESS --- */
        #nf-overlay {
            position: fixed; top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(255, 255, 255, 0.95); z-index: 99999;
            display: none; flex-direction: column; justify-content: center; align-items: center;
        }
        .nf-progress-box {
            background: var(--nb-lime);
            border: 5px solid #000;
            padding: 50px;
            width: 600px;
            max-width: 90%;
            border-radius: 25px;
            box-shadow: 15px 15px 0px #000;
            text-align: center;
            animation: popIn 0.3s ease-out;
        }
        .nf-progress-title {
            font-size: 32px; font-weight: 900; text-transform: uppercase;
            margin-bottom: 15px; color: #000;
        }
        .nf-bar-track {
            width: 100%; height: 50px;
            background: #fff;
            border: 4px solid #000;
            border-radius: 50px;
            overflow: hidden;
            position: relative;
            margin-bottom: 20px;
        }
        .nf-bar-fill {
            height: 100%;
            background: repeating-linear-gradient(45deg, #000, #000 10px, #333 10px, #333 20px);
            width: 0%;
            transition: width 0.4s ease-in-out;
        }

        /* --- TOAST --- */
        #nf-toast {
            visibility: hidden; min-width: 300px; background-color: #000; color: var(--nb-lime);
            text-align: center; border: 3px solid var(--nb-lime); border-radius: 12px; padding: 20px;
            position: fixed; z-index: 99999; right: 30px; top: 50px;
            box-shadow: 8px 8px 0px rgba(0,0,0,0.3); font-weight: bold; font-size: 18px;
            transform: translateX(120%); transition: transform 0.3s ease-in-out;
        }
        #nf-toast.show { visibility: visible; transform: translateX(0); }

        @keyframes popIn { 0% { transform: scale(0.8); opacity: 0; } 100% { transform: scale(1); opacity: 1; } }
    </style>
    <?php
}

// ==========================================================================
// 2. MENU & DASHBOARD PAGE SETUP
// ==========================================================================
add_action('admin_menu', 'nf_plugin_menu');

function nf_plugin_menu() {
    add_menu_page('BrainyPress', 'BrainyPress', 'manage_options', 'brainypress', 'nf_dashboard_page', 'dashicons-lightbulb', 6);
}

function nf_dashboard_page() {
    // ----------------------------
    // SAVE SETTINGS LOGIC (STRICT SECURITY FIX)
    // ----------------------------
    if (isset($_POST['nf_save_key'])) {
        // 🔥 Fix: wp_unslash first, then sanitize, then verify
        $nonce_val = isset($_POST['nf_settings_nonce']) ? sanitize_text_field(wp_unslash($_POST['nf_settings_nonce'])) : '';
        if ( ! wp_verify_nonce($nonce_val, 'nf_save_settings_action') ) {
             wp_die('Security Check Failed');
        }

        update_option('nf_gemini_api_key', isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '');
        update_option('nf_tw_consumer_key', isset($_POST['tw_consumer_key']) ? sanitize_text_field(wp_unslash($_POST['tw_consumer_key'])) : '');
        update_option('nf_tw_consumer_secret', isset($_POST['tw_consumer_secret']) ? sanitize_text_field(wp_unslash($_POST['tw_consumer_secret'])) : '');
        update_option('nf_tw_access_token', isset($_POST['tw_access_token']) ? sanitize_text_field(wp_unslash($_POST['tw_access_token'])) : '');
        update_option('nf_tw_access_secret', isset($_POST['tw_access_secret']) ? sanitize_text_field(wp_unslash($_POST['tw_access_secret'])) : '');
        
        echo '<script>window.onload = function() { showToast("✅ Settings Saved Successfully!"); }</script>';
    }

    // ----------------------------
    // CREATE ORDER LOGIC (STRICT SECURITY FIX)
    // ----------------------------
    if (isset($_POST['nf_create_order'])) {
        // 🔥 Fix: wp_unslash first
        $nonce_val = isset($_POST['nf_create_nonce']) ? sanitize_text_field(wp_unslash($_POST['nf_create_nonce'])) : '';
        if ( ! wp_verify_nonce($nonce_val, 'nf_create_order_action') ) {
             wp_die('Security Check Failed');
        }

        $orders = get_option('nf_active_orders', []);
        
        if (count($orders) >= 5) {
             echo '<script>window.onload = function() { showToast("❌ Limit Reached: Max 5 orders."); }</script>';
        } else {
            $new_order = [
                'id' => uniqid('nf_'),
                // 🔥 Fix: wp_unslash for all inputs
                'category_id' => isset($_POST['cat_id']) ? intval(wp_unslash($_POST['cat_id'])) : 0,
                'prompt' => isset($_POST['user_prompt']) ? sanitize_textarea_field(wp_unslash($_POST['user_prompt'])) : '',
                'schedule_time' => isset($_POST['schedule_time']) ? sanitize_text_field(wp_unslash($_POST['schedule_time'])) : '',
                'post_count' => isset($_POST['post_count']) ? intval(wp_unslash($_POST['post_count'])) : 1,
                'last_run' => '',
                'total_posts' => 0
            ];
            $orders[] = $new_order;
            update_option('nf_active_orders', $orders);
            echo '<script>window.onload = function() { showToast("🚀 Automation Created!"); window.history.replaceState(null, null, "admin.php?page=brainypress"); }</script>';
        }
    }

    // ----------------------------
    // DELETE ORDER LOGIC (STRICT SECURITY FIX)
    // ----------------------------
    if (isset($_GET['delete_order'])) {
        $nonce_val = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';
        $del_id = sanitize_text_field(wp_unslash($_GET['delete_order']));

        if (current_user_can('manage_options') && wp_verify_nonce($nonce_val, 'delete_order_' . $del_id)) {
            $orders = get_option('nf_active_orders', []);
            foreach ($orders as $key => $order) {
                if ($order['id'] == $del_id) {
                    unset($orders[$key]);
                }
            }
            update_option('nf_active_orders', $orders);
            echo '<script>window.onload = function() { showToast("🗑️ Order Deleted."); window.history.replaceState(null, null, "admin.php?page=brainypress"); }</script>';
        }
    }

    // Fetch Values (Escaped later in HTML)
    $api_key = get_option('nf_gemini_api_key', '');
    $orders = get_option('nf_active_orders', []);
    $tw_ck = get_option('nf_tw_consumer_key', '');
    $tw_cs = get_option('nf_tw_consumer_secret', '');
    $tw_at = get_option('nf_tw_access_token', '');
    $tw_as = get_option('nf_tw_access_secret', '');

    ?>
    <div class="nf-wrap">
        
        <div class="nf-header">
            <h1 class="nf-animated-title">BrainyPress 🧠 v2.7</h1>
            <p style="font-size: 20px; font-weight: 700; margin: 0; opacity: 0.9;">The AI Content Machine.</p>
            <div class="nf-slogan">Automate. Dominate. Monetize.</div>
            <div style="margin-top: 15px;">
                <span class="nf-slogan" style="background:#fff; color:#000; font-size:12px; box-shadow:none; border:2px solid #000;">✅ WordPress 6.7 Tested</span>
            </div>
        </div>

        <form method="post" action="admin.php?page=brainypress">
            <?php wp_nonce_field('nf_save_settings_action', 'nf_settings_nonce'); ?>
            
            <div class="nf-card">
                <h2 style="font-size: 28px; text-transform: uppercase; font-weight:900;">🔑 Step 1: Connect Brain (API)</h2>
                <div style="margin-bottom: 25px;">
                    <label style="display:block; margin-bottom:5px;"><strong>Google Gemini API Key</strong></label>
                    <input type="password" name="api_key" value="<?php echo esc_attr($api_key); ?>" class="nf-input" placeholder="Paste Gemini API Key here...">
                    <small>We use <strong>Google Search Grounding</strong> for 100% Latest Data.</small>
                </div>

                <div style="background: #e0f2fe; border: 3px dashed #000; padding: 25px; border-radius: 12px;">
                    <h3 style="margin-top:0;">🐦 Twitter (X) Auto-Post (Optional)</h3>
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                        <div>
                            <label>API Key (Consumer)</label>
                            <input type="text" name="tw_consumer_key" value="<?php echo esc_attr($tw_ck); ?>" class="nf-input">
                        </div>
                        <div>
                            <label>API Secret (Consumer)</label>
                            <input type="password" name="tw_consumer_secret" value="<?php echo esc_attr($tw_cs); ?>" class="nf-input">
                        </div>
                        <div>
                            <label>Access Token</label>
                            <input type="text" name="tw_access_token" value="<?php echo esc_attr($tw_at); ?>" class="nf-input">
                        </div>
                        <div>
                            <label>Access Secret</label>
                            <input type="password" name="tw_access_secret" value="<?php echo esc_attr($tw_as); ?>" class="nf-input">
                        </div>
                    </div>
                </div>

                <div style="margin-top: 30px; text-align:right;">
                    <button type="submit" name="nf_save_key" class="nf-btn nf-btn-primary">💾 Save All Settings</button>
                </div>
            </div>
        </form>

        <div class="nf-card" style="border-color: var(--nb-lime);">
            <h2 style="font-size: 28px; text-transform: uppercase; font-weight:900;">📝 Step 2: Create Automation</h2>
            <form method="post" action="admin.php?page=brainypress">
                <?php wp_nonce_field('nf_create_order_action', 'nf_create_nonce'); ?>

                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px;">
                    <div>
                        <label><strong>Target Category</strong></label>
                        <select name="cat_id" class="nf-select">
                            <option value="">Select Category</option>
                            <?php
                            $args = array('orderby' => 'name', 'order' => 'ASC', 'hide_empty' => false);
                            $categories = get_categories($args);
                            foreach( $categories as $category ) {
                                echo '<option value="' . esc_attr($category->term_id) . '">' . esc_html($category->name) . '</option>';
                            }
                            ?>
                        </select>
                    </div>
                    <div>
                        <label><strong>Posts Per Run</strong></label>
                        <select name="post_count" class="nf-select">
                            <option value="1">1 Post</option>
                            <option value="2">2 Posts</option>
                            <option value="3">3 Posts</option>
                            <option value="4">4 Posts</option>
                            <option value="5">5 Posts</option>
                        </select>
                    </div>
                </div>
                <br>
                <label><strong>AI Prompt (Instruction)</strong></label>
                <textarea name="user_prompt" class="nf-textarea" rows="4" placeholder="Example: Find latest trending news on Bitcoin and write a detailed analysis..."></textarea>
                
                <div style="margin-top: 20px; display:flex; align-items:center; justify-content:space-between;">
                    <div>
                        <label><strong>Schedule Time (Daily)</strong></label>
                        <input type="time" name="schedule_time" class="nf-input" style="width: auto; display: inline-block; margin:0;">
                    </div>
                    <button type="submit" name="nf_create_order" class="nf-btn nf-btn-accent">➕ Create Order</button>
                </div>
            </form>
        </div>

        <h2 style="font-size:28px; font-weight:900; margin-bottom:20px;">📊 Active Automations</h2>
        <table class="nf-table">
            <thead>
                <tr>
                    <th>Task / Prompt</th>
                    <th>Settings</th>
                    <th>Stats</th>
                    <th style="text-align: right;">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($orders)): ?>
                    <tr><td colspan="4" style="text-align:center; padding:40px;">No Active Orders. Create one above!</td></tr>
                <?php else: ?>
                    <?php foreach ($orders as $order): ?>
                    <tr id="row-<?php echo esc_attr($order['id']); ?>">
                        <td width="40%">
                            <strong style="font-size:18px; color:var(--nb-lime); background:#000; padding:2px 8px;"><?php echo esc_html(get_cat_name($order['category_id'])); ?></strong>
                            <p style="font-size: 0.9em; opacity: 0.8; margin-top:10px;"><?php echo esc_html(substr($order['prompt'], 0, 80)) . '...'; ?></p>
                        </td>
                        <td>
                            <div>🕒 <?php echo esc_html($order['schedule_time']); ?></div>
                            <div style="margin-top:5px;">📑 <strong><?php echo esc_html($order['post_count']); ?></strong> Post(s)</div>
                        </td>
                        <td>
                            <span style="background:#000; color:#fff; padding:8px 15px; border-radius:50px;">Total: <?php echo esc_html($order['total_posts']); ?></span>
                        </td>
                        <td style="text-align: right;">
                            <button onclick="runInstant('<?php echo esc_js($order['id']); ?>')" class="nf-btn nf-btn-primary" id="btn-<?php echo esc_attr($order['id']); ?>">
                                ⚡ Run
                            </button>
                            
                            <?php 
                            $del_url = wp_nonce_url("admin.php?page=brainypress&delete_order=" . $order['id'], 'delete_order_' . $order['id']); 
                            ?>
                            <a href="<?php echo esc_url($del_url); ?>" class="nf-btn nf-btn-danger" style="margin-left: 10px;">X</a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>

        <div class="nf-footer">
            <h3>Designed & Developed by <strong>Naveed Ali Shah</strong> ❤️</h3>
            <p>Built for high-performance publishers. v2.7 Stable</p>
            <div class="nf-footer-links">
                <a class="nf-footer-btn" onclick="openModal('use')">📖 How to Use</a>
                <a class="nf-footer-btn" onclick="openModal('privacy')">🔒 Privacy</a>
                <a class="nf-footer-btn" onclick="openModal('disclaimer')">⚠️ Disclaimer</a>
                <a class="nf-footer-btn" href="https://mrnaveedshah.com/contact" target="_blank">📞 Contact</a>
            </div>
        </div>

    </div>

    <div id="nf-info-modal">
        <div class="nf-modal-content">
            <span class="nf-close-modal" onclick="closeModal()">X</span>
            <div id="nf-modal-title" class="nf-modal-title"></div>
            <div id="nf-modal-body" class="nf-modal-body"></div>
        </div>
    </div>

    <div id="nf-overlay">
        <div class="nf-progress-box">
            <div class="nf-progress-title">Working Magic...</div>
            <div class="nf-progress-desc">Connecting to BrainyPress AI...</div>
            <div class="nf-bar-track">
                <div class="nf-bar-fill" id="nf-fill-bar"></div>
            </div>
            <p style="margin-top: 25px; font-weight: bold; font-size: 16px; color: #000;">⚠️ DO NOT CLOSE THIS WINDOW</p>
        </div>
    </div>

    <div id="nf-toast">Notification Message</div>

    <script>
    function showToast(message) {
        var x = document.getElementById("nf-toast");
        x.innerHTML = message;
        x.className = "show";
        setTimeout(function(){ x.className = x.className.replace("show", ""); }, 4000);
    }

    // --- MODAL CONTENT ---
    const modalData = {
        'use': {
            title: '⚡ How To Start',
            body: '<p><strong>Step 1: Get API Keys</strong><br>Go to Google AI Studio and get your free Gemini API Key. Paste it in Step 1.</p><p><strong>Step 2: Create Automation</strong><br>Select a Category (e.g., Crypto News).<br>Enter a Prompt: <em>"Find the latest Bitcoin news and write a detailed analysis."</em></p><p><strong>Step 3: Run</strong><br>Click <strong>"Run Now"</strong> to generate instantly, or set a Daily Time for auto-posting.</p><hr><p><strong>🐦 Twitter Integration:</strong><br>If you want auto-tweets, add your Twitter Developer Keys in Step 1. Ensure the app has "Read & Write" permissions.</p>'
        },
        'privacy': {
            title: '🔒 Privacy Policy',
            body: '<p><strong>Your Data is Safe.</strong></p><p>BrainyPress connects <strong>directly</strong> from your WordPress server to Google and Twitter APIs. We do not act as a middleman.</p><ul><li>Your API Keys are stored locally in your WordPress Database (`wp_options`).</li><li>We do not track your prompts or generated content.</li><li>You are in full control of your data.</li></ul>'
        },
        'disclaimer': {
            title: '⚠️ Important Disclaimer',
            body: '<p><strong>AI Responsibility</strong></p><p>This plugin uses Artificial Intelligence to generate content. While we use Google Grounding to fetch real-time facts, AI can sometimes hallucinate or provide inaccurate information.</p><p><strong>You are responsible</strong> for reviewing content before or after it is published. The developer (Naveed Ali Shah) is not liable for any content generated by this tool.</p>'
        }
    };

    function openModal(type) {
        if(modalData[type]) {
            document.getElementById('nf-modal-title').innerHTML = modalData[type].title;
            document.getElementById('nf-modal-body').innerHTML = modalData[type].body;
            document.getElementById('nf-info-modal').style.display = 'flex';
        }
    }

    function closeModal() {
        document.getElementById('nf-info-modal').style.display = 'none';
    }

    function runInstant(orderId) {
        var overlay = document.getElementById('nf-overlay');
        var bar = document.getElementById('nf-fill-bar');
        var title = document.querySelector('.nf-progress-title');
        var desc = document.querySelector('.nf-progress-desc');
        
        overlay.style.display = 'flex';
        bar.style.width = "5%";
        title.innerHTML = "🧠 Initializing...";

        // Emoji Progress
        var progress = 5;
        var interval = setInterval(function() {
            if (progress < 95) {
                progress += Math.floor(Math.random() * 8);
                if(progress > 95) progress = 95;
                bar.style.width = progress + "%";
                
                if(progress > 10 && progress < 40) { title.innerHTML = "🌍 Grounding..."; desc.innerHTML = "Searching Google for Real-Time Data..."; }
                if(progress > 40 && progress < 70) { title.innerHTML = "✍️ Writing..."; desc.innerHTML = "Drafting High-Quality Content..."; }
                if(progress > 70) { title.innerHTML = "🚀 Finalizing..."; desc.innerHTML = "Posting to WordPress & Twitter..."; }
            }
        }, 800);

        var data = {
            'action': 'nf_run_instant_action',
            'order_id': orderId,
            // 🔥 FIX: Escaping JS Nonce
            'nonce': '<?php echo esc_js(wp_create_nonce("nf_run_instant_nonce")); ?>'
        };

        jQuery.post(ajaxurl, data, function(response) {
            clearInterval(interval);
            
            if(response.success) {
                bar.style.width = "100%";
                title.innerHTML = "🎉 Success!";
                desc.innerHTML = response.data.message;
                setTimeout(function(){ 
                    overlay.style.display = 'none';
                    showToast("✅ " + response.data.message);
                    location.reload(); 
                }, 2000);
            } else {
                overlay.style.display = 'none';
                showToast("❌ " + response.data.message);
            }
        });
    }
    </script>
    <?php
}

// ==========================================================================
// 3. CORE LOGIC (HANDLERS)
// ==========================================================================

add_action('wp_ajax_nf_run_instant_action', 'nf_run_instant_handler');
add_action('nf_hourly_check_v8', 'nf_process_orders_v8'); 
add_action('wp', 'nf_setup_schedule_v8');

function nf_setup_schedule_v8() {
    if (!wp_next_scheduled('nf_hourly_check_v8')) {
        wp_schedule_event(current_time('timestamp'), 'hourly', 'nf_hourly_check_v8');
    }
}

function nf_run_instant_handler() {
    // 🔥 FIX: Sanitize + Verify Nonce
    $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
    if ( ! wp_verify_nonce($nonce, 'nf_run_instant_nonce') ) {
        wp_send_json_error(['message' => 'Security check failed']);
    }

$order_id = isset($_POST['order_id']) ? sanitize_text_field(wp_unslash($_POST['order_id'])) : '';
    nf_execute_post_logic($order_id, true);
}

function nf_process_orders_v8() {
    $orders = get_option('nf_active_orders', []);
    $current_hour = current_time('H');
    $today = current_time('Y-m-d');

    foreach ($orders as $order) {
        $order_hour = explode(':', $order['schedule_time'])[0];
        if ($order_hour === $current_hour && $order['last_run'] !== $today) {
            nf_execute_post_logic($order['id'], false);
        }
    }
}

// ==========================================================================
// 4. MASTER FUNCTION (EXECUTION)
// ==========================================================================
function nf_execute_post_logic($order_id, $is_ajax) {
    $api_key = get_option('nf_gemini_api_key');
    $orders = get_option('nf_active_orders', []);
    $target_order = null;
    $order_key = null;

    foreach ($orders as $key => $o) {
        if ($o['id'] == $order_id) { $target_order = $o; $order_key = $key; break; }
    }

    if (!$target_order || !$api_key) {
        if ($is_ajax) wp_send_json_error(['message' => 'Error: Order or API Key missing']); 
        return;
    }

    $count = intval($target_order['post_count']);
    $posts_made = 0;
    $last_msg = "";

    for ($i = 0; $i < $count; $i++) {
        $res = nf_generate_content_universal($api_key, wp_unslash($target_order['prompt']));
        
        if ($res['success']) {
            $content = $res['content'];
            $content = preg_replace('/^```html|```/i', '', $content);
            $content = trim($content);

            $title = '';
            if (preg_match('/<h1[^>]*>(.*?)<\/h1>/si', $content, $matches)) {
                $title = wp_strip_all_tags($matches[1]);
                $content = str_replace($matches[0], '', $content);
            } elseif (preg_match('/<h2[^>]*>(.*?)<\/h2>/si', $content, $matches)) {
                $chk = wp_strip_all_tags($matches[1]);
                if (strlen($chk) > 10) { $title = $chk; $content = str_replace($matches[0], '', $content); }
            }

            if (empty($title)) $title = 'Update: ' . current_time('M d, Y');
            $title = str_replace(['Title:', 'Subject:', '"'], '', $title);

            $post_id = wp_insert_post([
                'post_title'    => trim($title),
                'post_content'  => $content,
                'post_status'   => 'publish',
                'post_author'   => 1, 
                'post_category' => array($target_order['category_id'])
            ]);
            
            if ($post_id) {
                nf_post_to_twitter_api($post_id);
            }

            $posts_made++;
        } else {
            $last_msg = $res['error'];
        }
    }

    if ($posts_made > 0) {
        $orders[$order_key]['total_posts'] += $posts_made;
        $orders[$order_key]['last_run'] = current_time('Y-m-d');
        update_option('nf_active_orders', $orders);
        
        if ($is_ajax) wp_send_json_success(['message' => "Success! Posted $posts_made articles."]);
    } else {
        if ($is_ajax) wp_send_json_error(['message' => "Failed: $last_msg"]);
    }
}

// ==========================================================================
// 5. AI GENERATION (NO FAIL + SAFETY REMOVED + TOPIC INJECTION)
// ==========================================================================
function nf_generate_content_universal($api_key, $prompt) {
    $model = 'gemini-2.0-flash-exp'; 
    $today_date = current_time('l, F j, Y');
    
    // PHP Topic Injector
    $forced_topics = [
        "Market Sentiment Analysis & Whale Activity",
        "Technical Analysis of Top 3 Gainers Today",
        "Hidden Low-Cap Gems & Risk Analysis",
        "Major Exchange Listings & Liquidity Updates",
        "Regulatory News & Global Government Stance",
        "DeFi Protocols & Total Value Locked (TVL) Updates",
        "Memecoin Frenzy: Top Performers on Solana/Base",
        "Bitcoin ETF Flows & Institutional Buying",
        "Web3 Gaming & Metaverse Token Updates",
        "Layer 2 Scaling Solutions & Upgrades"
    ];
    $random_injection = $forced_topics[wp_rand(0, count($forced_topics) - 1)];

    $target_length = "1500";
    if (preg_match('/(\d+)\s*words/i', $prompt, $matches)) {
        $target_length = $matches[1];
    }

    $final_prompt = "
    SYSTEM ROLE: You are 'BrainyPress', a Senior Crypto Journalist & Analyst.
    CRITICAL CONTEXT:
    - User Prompt: '$prompt'
    - **MANDATORY FOCUS FOR THIS ARTICLE:** '$random_injection'
    - Date: $today_date
    INSTRUCTIONS:
    1. IGNORE generic summaries. Write a deep-dive article SPECIFICALLY about **$random_injection**.
    2. USE GOOGLE SEARCH to find the exact data for '$random_injection' happening right now (last 24 hours).
    3. LENGTH: The user requested **$target_length+ Words**. 
       - If news is short, expand with: Historical Context, Expert Opinions, Future Price Targets (2025-2030), and Pro Tips.
       - Do not stop writing until you reach the length.
    FORMAT (HTML ONLY - NO MARKDOWN):
    <h1>[Clickbait SEO Title related to $random_injection]</h1>
    <p>[Strong Hook Intro]</p>
    <h2>[Detailed Heading 1]</h2>
    <p>[Content...]</p>
    ...
    <h3>Conclusion & Outlook</h3>
    <p>[Final thoughts]</p>
    ";

    $url = "https://generativelanguage.googleapis.com/v1beta/models/$model:generateContent?key=" . $api_key;
    
    $body_array = [
        "contents" => [ ["parts" => [ ["text" => $final_prompt] ]] ],
        "tools" => [ [ "google_search" => (object)[] ] ],
        "safetySettings" => [
            [ "category" => "HARM_CATEGORY_HARASSMENT", "threshold" => "BLOCK_NONE" ],
            [ "category" => "HARM_CATEGORY_HATE_SPEECH", "threshold" => "BLOCK_NONE" ],
            [ "category" => "HARM_CATEGORY_SEXUALLY_EXPLICIT", "threshold" => "BLOCK_NONE" ],
            [ "category" => "HARM_CATEGORY_DANGEROUS_CONTENT", "threshold" => "BLOCK_NONE" ]
        ]
    ];
    
    $max_retries = 2;
    $attempt = 0;
    
    while ($attempt < $max_retries) {
        $attempt++;
        
        $response = wp_remote_post($url, [
            'body' => json_encode($body_array),
            'headers' => ['Content-Type' => 'application/json'],
            'timeout' => 120,
            'sslverify' => false
        ]);

        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            if (isset($data['candidates'][0]['content']['parts'][0]['text'])) {
                $generated_text = $data['candidates'][0]['content']['parts'][0]['text'];
                if (strlen($generated_text) > 200) {
                    return ['success' => true, 'content' => $generated_text];
                }
            }
        }
        sleep(1);
    }
    
    $error_msg = isset($data['error']['message']) ? $data['error']['message'] : "Unknown Error";
    return ['success' => false, 'error' => "Failed after 2 attempts. API Error: " . $error_msg];
}

// ==========================================================================
// 6. TWITTER API POSTING (OAUTH 1.0a)
// ==========================================================================
function nf_post_to_twitter_api($post_id) {
    $ck = get_option('nf_tw_consumer_key');
    $cs = get_option('nf_tw_consumer_secret');
    $at = get_option('nf_tw_access_token');
    $as = get_option('nf_tw_access_secret');

    if (!$ck || !$cs || !$at || !$as) return;

    $text = "🔥 " . get_the_title($post_id) . "\n\n👇 Read More:\n" . get_permalink($post_id) . "\n\n#Crypto #News #Bitcoin";
    $api_url = 'https://api.twitter.com/2/tweets';

    $oauth = [
        'oauth_consumer_key' => $ck,
        'oauth_nonce' => md5(microtime() . wp_rand()),
        'oauth_signature_method' => 'HMAC-SHA1',
        'oauth_timestamp' => time(),
        'oauth_token' => $at,
        'oauth_version' => '1.0'
    ];

    $base_string = 'POST&' . rawurlencode($api_url) . '&' . rawurlencode(nf_build_query($oauth));
    $signing_key = rawurlencode($cs) . '&' . rawurlencode($as);
    $signature = base64_encode(hash_hmac('sha1', $base_string, $signing_key, true));
    $oauth['oauth_signature'] = $signature;

    $header_parts = [];
    foreach($oauth as $k => $v) { $header_parts[] = $k . '="' . rawurlencode($v) . '"'; }
    $auth_header = 'OAuth ' . implode(', ', $header_parts);

    wp_remote_post($api_url, [
        'headers' => [
            'Authorization' => $auth_header,
            'Content-Type' => 'application/json'
        ],
        'body' => json_encode(['text' => $text]),
        'method' => 'POST',
        'blocking' => true
    ]);
}

function nf_build_query($params) {
    ksort($params);
    $out = [];
    foreach($params as $k => $v) { $out[] = rawurlencode($k) . '=' . rawurlencode($v); }
    return implode('&', $out);
}
?>